/*

Alternatives: Method-calls and event-switchboarding

.

ADVANTAGES of using simple method-calls:

 --- Simple, fast because of little programming and compiling overhead.

 --- Direct contact and insight into the communicating entities.

DISadvantages of using simple method-calls:

 --- The calling-class or -object has to be aware of the specific methods,

     which the called entity uses (the second item of the advantages).

 --- Usually only one target entity per method-call available.

.

ADVANTAGES of using event-objects with software-switchboarding:

 --- The communicating entities may only be coupled by the common knowledge

     of the message-sending protocol (another form of encapsulation).

 --- A single event-object dispatched notifies multiple targets.

 --- Full freedom of implementing multicast multisource message passing, for example in

     instantiating a switchboard- or adapter-object, that routes events to their handlers.

 --- One of the real advantages of the event-model shows when multiple sources

     produce multiple event-objects whose references have to be channeled properly to multiple drains.

DISAdvantages of using event-objects with software-switchboarding:

 --- Large overhead of implementation of message-sending protocol for the infrastructure-programmer.

 --- Additional learning for the application-programmer.

 --- The programming overhead also shows in added consumption of processor-time.

*/

/* So a simple unisource unicast message or event passing

   usually can be substituted by a method-call.

The mechanism becomes beneficial in multisource multicast situations and some other circumstances:

-- Changes in design or existence of sources and drains of messages should be possible.

-- Registering and de-registering of events should be easy and localized at one point.

-- Registering should should be done with sources and drains instead of messages and handlers.

 (Hides the communication implementation.)

 That is possible by using an adapter- or switchboard-object as a registry-center

 for source- and drain-objects.

 This adapter- or switchboard-object receives the event-objects from the registered sources

 and forwards them to the appropriate handlers in the drains.

 If there is only one drain-object,

 then the adapter- or switchboard-object often is attached to the drain

 by defining it as an inner class.

*/

 // EVENT

 // The class EventObject has

 // --- a protected (transient(not part of serialization)) member-variable the Object "source",

 // --- a constructor with the Object source as a parameter,

 // --- and the methods public Object getSource() and public String toString()

 // An extension of this class may introduce additional variables that transport data

 // (even a reference to a Thread-object).

 // An extension may define methods

 // that transform data and by their association to the event-object

 // are available to source AND drain.

 // EVENT

class NumberEvent extends java.util.EventObject

{ static int noOfEventInstantiation=0;

  int noOfEvent=0;

  int info;

  NumberEvent(Object source, int n)

  { super(source);

   noOfEventInstantiation++;

   noOfEvent=noOfEventInstantiation;

   info=n;

   System.out.println( source.getClass().getName() + "  instantiated the event " +

                                                 noOfEvent + " with the number-data=" + info );

  }

}

//NumberEvent(){super(this.getClass().getDeclaringClass());}

// can't reference this before the super-class constructor has been called}

// end NumberEvent

/* The NumberEvent-class can count its instances

   with the static integer-variable noOfEventInstantiation.

   To identify the event each instance of NumberEvent gets an instance-variable noOfEvent

   with the current number. The instance-variable 'info' holds the data of the NumberEvent-object.

   This event-class is extended:

*/

class TextNumberEvent extends NumberEvent

  { String messageText;

    TextNumberEvent( Object source, int n, String s )

    { super(source,n);

      messageText = s;

      System.out.println( source.getClass().getName()+ " and contains the text="+s );

}} // end TextNumberEvent

////////////////////

/*

Now the programmer introduces an interface EventDispatcher,

 this requires each class that implements EventDispatcher to implement two methods:

  addEventListener() and removeEventListener().

  Then two different kinds of sources are introduced,

  --- one, SourceA that accesses the relay() method of a Switchboard object directly;

  --- the other, SourceB sends an event just by calling the

      dispatchTo(Switchboard, java.util.EventObject) method.

*/

////////////////////// SOURCES

interface EventDispatcher

{ void addEventListener( Switchboard l );

  void removeEventListener( Switchboard l );

}

// that interface tags sources and furnishes the names for standard methods

class SourceA implements EventDispatcher

{ SourceA(){}

  Switchboard listener = null;

  public void addEventListener( Switchboard l ){ listener=l;};

  public void removeEventListener( Switchboard l ){ listener=null;};

  void activate(){ listener.relay( new NumberEvent(this,5) ); }

}

class SourceB implements EventDispatcher

{ SourceB(){}

  Switchboard listener = null;

  public void addEventListener( Switchboard l ){ listener=l;}

  public void removeEventListener( Switchboard l ){ listener=null;}

  void dispatchTo(Switchboard l, java.util.EventObject evt){l.relay(evt);}

  void activate()

  { dispatchTo( listener, new NumberEvent(this,8) );

    dispatchTo( listener, new TextNumberEvent(this,9,"Mail!") );

  }

}

///////////////////// DRAINS

/* There are two drain-classes:

   DrainX has two event-handlers that each take a different event-class and

   have different names. The drain-class DrainY has one event-handling method.

*/

class DrainX

{ DrainX(){};

  public synchronized void workoverEvent( NumberEvent evt )

  { System.out.println( this.getClass().getName() + " has received event " + evt.noOfEvent +

                                                                  " with number=" + evt.info );

  }

  public synchronized void handleEvent( TextNumberEvent evt )

  { System.out.println( this.getClass().getName() + " has received event " +

                        evt.noOfEvent + evt.getClass().getName() + " with number=" + evt.info +

                        " text=" + evt.messageText );

   };

}

class DrainY

{ public synchronized void handleEvent( NumberEvent evt )

  { System.out.println( this.getClass().getName() + " has received event " + evt.noOfEvent +

                                                                      " with: " + (evt.info) );

  };

}// what if static?

class DrainZ

{ static void EventHandler( TextNumberEvent evt )

  { System.out.println

    ( DrainZ.class.getName() + " a static handler has received event " +

      evt.noOfEvent + " with: " + (evt.info) + "OOOOOOOOOOOOOO" );

};}

//////////////////  Switchboard

/* The Switchboard-class is the most complex part of this application:

The switchboard routes the event-objects from the sources to their appropriate drains.

The routing-mechanism is programmed by using Vector- and Hashtable-classes:

Arrays are found where a sequence of objects can be indexed by a natural number.

The java.util.Vector-class is used if the length of the sequence should be variable.

The java.util.Hashtable-class is used were variably many objects may be indexable by

 references to other objects.

Here the hashtable is used as the center of the switching-process:

The indices (or keys) are event types and the objects are vectors of "objectnames.methods"

 that are handlers of these event types.

The central method constructConnectionTableFromDrains(Object [] drains) does this work.



Here the java.lang.Class and the java.lang.reflect.* classes

 are used extensively to analyze objects and their members during run-time.

( For a more detailed discussion of the java.lang.reflect package and the functionalities

  of the java.lang.Class-class refer to Java Language Specification.)

*/

// Switchboard

class Switchboard implements java.util.EventListener

{ protected void registerSources(EventDispatcher[] sources)

  { for( int i = 0; i < sources.length; i++ ){ sources[i].addEventListener(this); }; // source-registering-process

  }



  protected java.util.Hashtable connections = new java.util.Hashtable(10,(float).5);

  // capacity, load factor(0.0=fast 1.0=small)



  protected java.util.Hashtable constructConnectionTableFromDrains(Object[] drains)

  { java.util.Hashtable table = new java.util.Hashtable();

    // keys are Event types, values are vectors with the objectnames.methods

    for( int i=0; i < drains.length; i++ )

    { java.lang.reflect.Method [] methodInfoArray = drains[i].getClass().getMethods();

      for( int k=0; k < methodInfoArray.length; k++ )

      { if( methodInfoArray[k].getParameterTypes().length == 1 )

        { // only one parameter (extensible to testing on modifier "void")

          boolean argumentIsEvent = false;

          Class param = methodInfoArray[k].getParameterTypes()[0];

          Class check = param;

          while( check != null && argumentIsEvent == false )

          { // that parameter is of type sub-class of EventObject

            if( check == java.util.EventObject.class )

            { argumentIsEvent = true;}else{ check = check.getSuperclass(); };

          }; // loop avoidable if there were a class Class method isSuperclass()

          if( argumentIsEvent )

          { Object objectref = drains[i];

            java.lang.reflect.Method method = methodInfoArray[k];

            java.util.Vector vect;

            if ( table.containsKey(param) )

            { vect = (java.util.Vector)table.get( param );}

            else{ vect = new java.util.Vector(); };

            Object [] pair = new Object[]{ objectref, method }; // can be anonymized

            vect.addElement(pair);

            table.put( param, vect );

          }; // end if argument of method is event

       }; // end if method only one parameter

      }; // end for methods count

    }; // end for drains count

    // testing before returning

    //

    //System.out.println(table.toString());

    //for(java.util.Enumeration e=table.keys();e.hasMoreElements();)

    // {Object[] info={table.get(e.nextElement())};System.out.println("   "+info);};

    return table;

    }

    void relay( java.util.EventObject evt )

    {

      // the event reference is the parameter of the method,

      // but the event-class is the argument of the hashtable!

      Object [] param = { evt };

      java.util.Vector targets = (java.util.Vector)connections.get( evt.getClass() );

      // select the target references and their handling methods

      for( int i=0; i < targets.size(); i++ )

      { // step through each target

        Object [] objrefMethodPair =  (Object [])targets.elementAt( i );

        // array caster: (type []) - array-initializer: {...}

        Object objectref = objrefMethodPair[0];

        java.lang.reflect.Method method = (java.lang.reflect.Method)objrefMethodPair[1];

        try{ method.invoke( objectref, param ); } // relay event to target handling method

        catch( java.lang.reflect.InvocationTargetException e ){ e.printStackTrace(); }

        catch(java.lang.IllegalAccessException e){ System.out.println(e); } // unnecessary

      }; // end for

     } // end relay

     Switchboard( EventDispatcher[] sources, Object[] drains )

     {  registerSources(sources);

        connections = constructConnectionTableFromDrains( drains );

        // now relay can route the events to their receiver(s)

     } // end Switchboard constructor

} // end Switchboard-class

// if only one global switchboard useful then use static methods and make it uninstantiable

// by making the default constructor private

public class RoutingDemo

{ public static void main( String args[] )

  { System.out.println( "\n\nRouting Demonstration" );

    SourceA sourceA = new SourceA();

    SourceB sourceB = new SourceB();

    DrainX drainX = new DrainX();

    DrainY drainY = new DrainY();// or static? does not work

    Switchboard router =

    new Switchboard( new EventDispatcher[] {sourceA, sourceB},

                              new Object[] {drainX, drainY, DrainZ.class} );

    sourceA.activate();

    sourceB.activate();

    sourceA.activate();

  } // end main

} // end RoutingDemo


















/*
[localhost:~/JavaCodeFiles/Appendix] marycosway% javac RoutingDemo.java
[localhost:~/JavaCodeFiles/Appendix] marycosway% java RoutingDemo


Routing Demonstration
SourceA instantiated the event 1 with the number-data=5
DrainX has received event 1 with number=5
DrainY has received event 1 with: 5
SourceB instantiated the event 2 with the number-data=8
DrainX has received event 2 with number=8
DrainY has received event 2 with: 8
SourceB instantiated the event 3 with the number-data=9
SourceB and contains the text=Mail!
DrainX has received event3TextNumberEvent with number=9 text=Mail!
SourceA instantiated the event 4 with the number-data=5
DrainX has received event 4 with number=5
DrainY has received event 4 with: 5
[localhost:~/JavaCodeFiles/Appendix] marycosway%
*/

/*
marycosway@linux:~/JavaCodeFiles/Appendix > javac RoutingDemo.java
marycosway@linux:~/JavaCodeFiles/Appendix > java RoutingDemo


Routing Demonstration
SourceA instantiated the event 1 with the number-data=5
DrainX has received event 1 with number=5
DrainY has received event 1 with: 5
SourceB instantiated the event 2 with the number-data=8
DrainX has received event 2 with number=8
DrainY has received event 2 with: 8
SourceB instantiated the event 3 with the number-data=9
SourceB and contains the text=Mail!
DrainX has received event3TextNumberEvent with number=9 text=Mail!
SourceA instantiated the event 4 with the number-data=5
DrainX has received event 4 with number=5
DrainY has received event 4 with: 5
marycosway@linux:~/JavaCodeFiles/Appendix >
*/


/*
C:\JavaCodeFiles\Appendix>javac RoutingDemo.java

C:\JavaCodeFiles\Appendix>java RoutingDemo


Routing Demonstration
SourceA instantiated the event 1 with the number-data=5
DrainX has received event 1 with number=5
DrainY has received event 1 with: 5
SourceB instantiated the event 2 with the number-data=8
DrainX has received event 2 with number=8
DrainY has received event 2 with: 8
SourceB instantiated the event 3 with the number-data=9
SourceB and contains the text=Mail!
DrainX has received event3TextNumberEvent with number=9 text=Mail!
SourceA instantiated the event 4 with the number-data=5
DrainX has received event 4 with number=5
DrainY has received event 4 with: 5

C:\JavaCodeFiles\Appendix>
*/


/* See copyright and legal notes in the book "Charting Java", hinted at in the text-file: file:///<parentPath>/JavaCodeFiles/BASIC_INFORMATION.txt */